@extends('partials.layouts.master')

@section('title', 'Waste Bags Management - MediTrack')
@section('meta_description', 'Track and manage medical waste bags. Monitor waste bag status, view creation and delivery timestamps, and generate disposal summaries.')
@section('title-sub', 'Waste bags')

@section('pagetitle')
<div class="d-flex justify-content-between align-items-center mb-3">
    <div class="flex items-center space-x-2 text-sm font-medium">
        <span class="main-home"><a href="{{ route('dashboard') }}" class="main-home">Home</a></span>
        <span class="main-home">/</span>
        <span class="main-home"><a href="{{ route('clinics') }}" class="main-home">Clinics</a></span>
        <span class="main-home">/</span>
        <span class="main-home"><a href="{{ route('moduls', ['clinic_id' => $clinic->id]) }}" class="main-home">Moduls</a></span>
        <span class="main-home">/</span>
        <span class="main-home"><a href="{{ route('Medical-Waste', ['clinic_id' => $clinic->id]) }}" class="main-home">Medical Waste</a></span>
        <span class="main-home">/</span>
        <span class="main-clinics">Waste bags</span>
    </div>

    <!-- Export CSV Button -->
    <div class="d-flex justify-content-end">
        <button class="btn px-5 fw-bold" style="border: 1px solid #00000034; color: #000;">
            Export CSV
        </button>
    </div>
</div>
@endsection

@section('content')

<div class="container-fluid p-4">
    <!-- Table Section -->
    <div class="border border-1 p-5 rounded-4 bg-white">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <h4 class="main-waste-bag mb-0">Waste Bags</h4>

            <div class="d-flex gap-2">
                <!-- Search -->
                <form method="GET" action="{{ route('Waste-bags', ['clinic_id' => $clinic->id]) }}" class="d-flex">
                    <div class="input-group" style="max-width: 280px; border: 1px solid #dee2e6; border-radius: 12px; overflow: hidden;">
                        <span class="input-group-text bg-white" style="border: none;">
                            <i class="bi bi-search text-muted"></i>
                        </span>
                        <input type="text" name="search" class="form-control ps-0" placeholder="Search"
                               style="border: none; height: 40px;" value="{{ request('search') }}">
                    </div>
                </form>
            </div>
        </div>

        <!-- Table -->
        <div class="card border-0 p-2">
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table align-middle mb-0">
                        <thead class="table-light">
                            <tr>
                                <th class="border-0 py-3">Clinic Name</th>
                                <th class="border-0 py-3">Waste Type</th>
                                <th class="border-0 py-3">Created At</th>
                                <th class="border-0 py-3">Closed At</th>
                                <th class="border-0 py-3">Delivered At</th>
                                <th class="border-0 py-3">Status</th>
                                <th class="border-0 py-3">Summary</th>
                            </tr>
                        </thead>
                        <tbody>
                            @forelse($wasteBags as $wasteBag)
                            <tr>
                                <td class="py-3">{{ $clinic->name }}</td>
                                <td class="py-3">{{ $wasteBag->wasteType->name ?? 'N/A' }}</td>
                                <td class="py-3">{{ $wasteBag->created_timestamp ? \Carbon\Carbon::parse($wasteBag->created_timestamp)->format('M d, Y') : 'N/A' }}</td>
                                <td class="py-3">{{ $wasteBag->closed_timestamp ? \Carbon\Carbon::parse($wasteBag->closed_timestamp)->format('M d, Y') : 'N/A' }}</td>
                                <td class="py-3">{{ $wasteBag->delivered_timestamp ? \Carbon\Carbon::parse($wasteBag->delivered_timestamp)->format('M d, Y') : 'N/A' }}</td>
                                <td class="py-3">
                                    <select class="form-select form-select-sm status-select"
                                            data-bag-id="{{ $wasteBag->id }}"
                                            data-current-status="{{ $wasteBag->status }}"
                                            style="width: 140px; cursor: pointer;">
                                        <option value="1" {{ $wasteBag->status == 1 ? 'selected' : '' }}>Active</option>
                                        <option value="2" {{ $wasteBag->status == 2 ? 'selected' : '' }}>Closed</option>
                                        <option value="3" {{ $wasteBag->status == 3 ? 'selected' : '' }}>Delivered</option>
                                    </select>
                                </td>
                                <td>
                                    <a href="{{ route('summary', ['clinic_id' => $clinic->id, 'waste_bag_id' => $wasteBag->id]) }}" class="btn btn-sm" style="background-color: #f0fbff; border: 1px solid #dee2e6;">
                                        <img src="{{ asset('assets/images/file.png') }}" alt="Summary" style="width: 20px; height: 20px;">
                                    </a>
                                </td>
                            </tr>
                            @empty
                            <tr>
                                <td colspan="7" class="text-center text-muted py-4">No waste bags found. Add your first waste bag!</td>
                            </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <div class="d-flex justify-content-between align-items-center mt-3 px-3">
                    <div class="text-muted">
                        Showing {{ $wasteBags->firstItem() ?? 0 }} to {{ $wasteBags->lastItem() ?? 0 }} of {{ $wasteBags->total() }} results
                    </div>

                    <div class="d-flex gap-2 align-items-center">
                        {{ $wasteBags->links('pagination::bootstrap-5') }}
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Success/Failed Dialog Modal -->
<div id="statusDialog" class="modal fade" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content" style="border-radius: 16px; border: none;">
            <div class="modal-body text-center p-4">
                <div id="dialogIcon" class="mb-3"></div>
                <h5 id="dialogTitle" class="mb-2"></h5>
                <p id="dialogMessage" class="text-muted mb-4"></p>
                <button type="button" class="btn px-4" id="dialogButton" data-bs-dismiss="modal"></button>
            </div>
        </div>
    </div>
</div>

<style>
    body {
        background-color: white !important;
    }

    .main-home {
        color: var(--Greyscale-900, #818898) !important;
        font-family: "Inter Tight";
        font-size: 14px;
        font-style: normal;
        font-weight: 500;
        line-height: 150%;
        letter-spacing: 0.28px;
    }

    .main-clinics {
        color: var(--Greyscale-400, #0D0D12) !important;
        font-family: "Inter Tight";
        font-size: 14px;
        font-style: normal;
        font-weight: 500;
        line-height: 150%;
        letter-spacing: 0.28px;
    }

    .main-waste-bag {
        color: var(--Greyscale-900, #0D0D12);
        font-family: "Inter Tight";
        font-size: 16px;
        font-style: normal;
        font-weight: 600;
        line-height: 150%;
        letter-spacing: 0.32px;
    }

    .status-badge {
        display: inline-flex;
        align-items: center;
        gap: 8px;
        padding: 6px 12px;
        border-radius: 20px;
        border: 1px solid #e0e0e0;
        background-color: #ffffff;
        font-size: 14px;
        font-weight: 500;
        width: fit-content;
    }

    .status-dot {
        display: inline-block;
        width: 8px;
        height: 8px;
        border-radius: 50%;
        flex-shrink: 0;
    }

    /* ACTIVE */
    .status-badge.active .status-dot {
        background-color: #28a745 !important;
    }
    .status-badge.active {
        color: #28a745;
    }

    /* CLOSED */
    .status-badge.closed .status-dot {
        background-color: #dc3545 !important;
    }
    .status-badge.closed {
        color: #dc3545;
    }

    /* DELIVERED */
    .status-badge.delivered .status-dot {
        background-color: #0d6efd !important;
    }
    .status-badge.delivered {
        color: #0d6efd;
    }

    .table-responsive {
        border-radius: 12px !important;
        overflow: hidden !important;
    }

    .table {
        border-collapse: separate !important;
        border-spacing: 0 !important;
    }

    .table thead tr:first-child th:first-child {
        border-top-left-radius: 12px !important;
    }

    .table thead tr:first-child th:last-child {
        border-top-right-radius: 12px !important;
    }

    .table tbody tr:last-child td:first-child {
        border-bottom-left-radius: 12px !important;
    }

    .table tbody tr:last-child td:last-child {
        border-bottom-right-radius: 12px !important;
    }

    .table thead th {
        border-bottom-width: 1px !important;
    }

    .table tbody td {
        border-bottom: 1px solid #dee2e6;
    }

    .table tbody tr:last-child td {
        border-bottom: none !important;
    }

    /* Status Select Styling */
    .status-select {
        border: 1px solid #e0e0e0;
        border-radius: 8px;
        padding: 6px 12px;
        font-size: 14px;
        font-weight: 500;
    }

    .status-select option[value="1"] {
        color: #28a745;
    }

    .status-select option[value="2"] {
        color: #dc3545;
    }

    .status-select option[value="3"] {
        color: #0d6efd;
    }

    /* Dialog Modal Styles */
    #statusDialog .modal-content {
        box-shadow: 0 10px 40px rgba(0, 0, 0, 0.15);
    }

    .dialog-icon-success {
        width: 60px;
        height: 60px;
        border-radius: 50%;
        background-color: #d4edda;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        font-size: 30px;
        color: #28a745;
    }

    .dialog-icon-error {
        width: 60px;
        height: 60px;
        border-radius: 50%;
        background-color: #f8d7da;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        font-size: 30px;
        color: #dc3545;
    }

    .dialog-btn-success {
        background: linear-gradient(180deg, #4598F9 0%, #3A77BF 100%);
        color: white;
        border: none;
        border-radius: 8px;
        padding: 10px 30px;
        font-weight: 500;
    }

    .dialog-btn-success:hover {
        background: linear-gradient(180deg, #3A77BF 0%, #2d5f99 100%);
        color: white;
    }

    .dialog-btn-error {
        background-color: #dc3545;
        color: white;
        border: none;
        border-radius: 8px;
        padding: 10px 30px;
        font-weight: 500;
    }

    .dialog-btn-error:hover {
        background-color: #c82333;
        color: white;
    }
</style>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const clinicId = {{ $clinic->id }};
    const csrfToken = '{{ csrf_token() }}';

    // Function to show custom dialog
    function showDialog(type, title, message) {
        const dialogIcon = document.getElementById('dialogIcon');
        const dialogTitle = document.getElementById('dialogTitle');
        const dialogMessage = document.getElementById('dialogMessage');
        const dialogButton = document.getElementById('dialogButton');

        if (type === 'success') {
            dialogIcon.innerHTML = '<div class="dialog-icon-success">✓</div>';
            dialogTitle.textContent = title;
            dialogMessage.textContent = message;
            dialogButton.textContent = 'OK';
            dialogButton.className = 'btn px-4 dialog-btn-success';
        } else {
            dialogIcon.innerHTML = '<div class="dialog-icon-error">✕</div>';
            dialogTitle.textContent = title;
            dialogMessage.textContent = message;
            dialogButton.textContent = 'Try Again';
            dialogButton.className = 'btn px-4 dialog-btn-error';
        }

        const modal = new bootstrap.Modal(document.getElementById('statusDialog'));
        modal.show();
    }

    // Handle status change
    document.querySelectorAll('.status-select').forEach(select => {
        select.addEventListener('change', function() {
            const bagId = this.getAttribute('data-bag-id');
            const currentStatus = this.getAttribute('data-current-status');
            const newStatus = this.value;

            if (newStatus === currentStatus) {
                return; // No change
            }

            // Confirm status change
            const statusNames = {1: 'Active', 2: 'Closed', 3: 'Delivered'};
            if (!confirm(`Are you sure you want to change status to ${statusNames[newStatus]}?`)) {
                this.value = currentStatus; // Revert selection
                return;
            }

            // Send AJAX request to update status
            fetch(`/moduls/${clinicId}/medical-waste/waste-bags/${bagId}/status`, {
                method: 'PUT',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': csrfToken,
                    'Accept': 'application/json'
                },
                body: JSON.stringify({
                    status: newStatus
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Update the data attribute
                    this.setAttribute('data-current-status', newStatus);

                    // Show success dialog
                    showDialog('success', 'Success!', data.message || 'Status updated successfully!');

                    // Reload page after dialog is closed
                    document.getElementById('statusDialog').addEventListener('hidden.bs.modal', function() {
                        window.location.reload();
                    }, { once: true });
                } else {
                    showDialog('error', 'Failed!', data.message || 'Failed to update status');
                    this.value = currentStatus; // Revert selection
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showDialog('error', 'Error!', 'An error occurred while updating status');
                this.value = currentStatus; // Revert selection
            });
        });
    });
});
</script>

@endsection
