@extends('partials.layouts.master')

@section('title', 'Analytics | FabKin Admin & Dashboards Template')
@section('title-sub', 'Dashboard')

@section('pagetitle')
<div x-data @click.outside="showModal = false" class="d-flex justify-content-between align-items-center mb-3">
    <div class="flex items-center space-x-2 text-sm font-medium">
        <span class="text-gray-500 hover:text-gray-700 transition duration-150">Home</span>
        <span class="text-gray-400">/</span>
        <span class="text-gray-500 hover:text-gray-700 transition duration-150">Clinics</span>
        <span class="text-gray-400">/</span>
        <span class="text-gray-500 hover:text-gray-700 transition duration-150">Moduls</span>
        <span class="text-gray-400">/</span>
        <span class="text-gray-500 hover:text-gray-700 transition duration-150">Medical Waste</span>
        <span class="text-gray-400">/</span>
        <span class="text-gray-900 font-bold">Collectors</span>
    </div>
    <div class="d-flex">
        <a href="#" 
           class="btn px-5 me-2 fw-bold" 
           style="border: 1px solid #00000034; color: #000;" 
           @click.prevent="exportCSV()">
           Export CSV
        </a>
        <a href="#" 
           class="btn text-white px-5" 
           style="background-color: #4598F9;" 
           @click.prevent="openAddModal()">
           Add Collector
        </a>
    </div>
</div>
@endsection

@section('content')

<script defer src="https://cdn.jsdelivr.net/npm/alpinejs@3.x.x/dist/cdn.min.js"></script>

<div class="container-fluid"
     x-data="collectorsManager">

    <div class="border border-2 p-4 bg-white" style="border-radius: 6px;">
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h4 class="fw-bold mb-0">All Collectors</h4>
            </div>
            <div class="d-flex gap-2">
                <!-- Search Input -->
                <div class="input-group" style="max-width: 280px; border: 1px solid #dee2e6; border-radius: 12px; overflow: hidden;">
                    <span class="input-group-text bg-white" style="border: none; border-radius: 20px 0 0 20px;">
                        <i class="bi bi-search text-muted"></i>
                    </span>
                    <input 
                        type="text" 
                        class="form-control ps-0" 
                        placeholder="Search" 
                        x-model="searchQuery"
                        style="border: none; height: 40px; outline: none; box-shadow: none;">
                </div>
                
                <!-- Filter Button -->
                <button class="btn btn-light filter bg-white d-flex align-items-center gap-2" 
                        @click="showFilterModal = true"
                        style="border-radius: 12px; padding: 8px 20px; border: 1px solid #e0e0e0; width:100px; height:40px;">
                    <i class="bi bi-funnel"></i>
                    <span>Filter</span>
                </button>
                
                <!-- Sort By Button -->
                <button class="btn btn-light sort bg-white d-flex align-items-center gap-2" 
                        @click="toggleSort()"
                        style="border-radius: 12px; padding: 8px 20px; border: 1px solid #e0e0e0; width:140px; height:40px;">
                    <i class="bi bi-arrow-down-up"></i>
                    <span>Sort by</span>
                </button>
            </div>
        </div>

        <!-- Collectors Table -->
        <div class="card border-0">
            <div class="card-body p-0">
                <div class="table-responsive">
                    <table class="table align-middle mb-0">
                        <thead class="table-light">
                            <tr>
                                <th class="py-3">Collector Name</th>
                                <th class="border-0 py-3">Location</th>
                                <th class="border-0 py-3">Collected bags</th>
                                <th class="border-0 py-3">Phone Number</th>
                                <th class="border-0 py-3 text-end">Bag History</th>
                                <th class="border-0 py-3 text-end">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <template x-if="filteredCollectors.length === 0">
                                <tr>
                                    <td colspan="6" class="text-center text-muted py-4">No matching collectors found.</td>
                                </tr>
                            </template>

                            <template x-for="(collector, index) in paginatedCollectors" :key="index">
                                <tr>
                                    <td class="py-3" x-text="collector.name"></td>
                                    <td class="py-3" x-text="collector.location"></td>
                                    <td class="py-3" x-text="collector.collectedBags"></td>
                                    <td class="py-3" x-text="collector.phoneNumber"></td>
                                    <td class="text-end py-3">
                                        <button class="btn btn-sm text-white" 
                                                @click="viewBagHistory(index)"
                                                style="background-color: #4598F9; min-width: 100px;">
                                            Bag History
                                        </button>
                                    </td>
                                    <td class="text-end py-3">
                                        <div class="d-inline-flex align-items-center">
                                            <button class="btn btn-sm me-1" 
                                                    @click="openEditModal(index)"
                                                    style="background-color: #f0fbff; border: 1px solid #dee2e6;">
                                                <i class="bi bi-pencil" style="color: #33cfff;"></i>
                                            </button>
                                            <button class="btn btn-sm" 
                                                    @click="deleteCollector(index)"
                                                    style="background-color: #fff0f3; border: 1px solid #dee2e6;">
                                                <i class="bi bi-trash" style="color: red;"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            </template>
                        </tbody>
                    </table>
                </div>

                <!-- Pagination -->
                <div class="d-flex justify-content-between align-items-center mt-4 px-3 border-top pt-3">
                    <div class="text-muted">
                        Showing <span x-text="startIndex + 1"></span> to <span x-text="endIndex"></span> of <span x-text="'500'"></span> results
                    </div>
                    
                    <div class="d-flex gap-3 align-items-center">
                        <!-- Per page dropdown -->
                        <div class="text-muted">Per page</div>
                        <select class="form-select form-select-sm" 
                                style="width: 70px; border-radius: 8px;" 
                                x-model="itemsPerPage" 
                                @change="resetPagination">
                            <option value="10">10</option>
                            <option value="25">25</option>
                            <option value="50">50</option>
                            <option value="100">100</option>
                        </select>
                        
                        <!-- Pagination buttons -->
                        <div class="pagination-container d-flex align-items-center">
                            <button class="pagination-btn prev-btn" 
                                    @click="prevPage" 
                                    :disabled="currentPage === 1">
                                <i class="bi bi-chevron-left"></i>
                            </button>
                            
                            <template x-for="page in visiblePages" :key="page">
                                <button class="pagination-btn" 
                                        :class="page === currentPage ? 'active' : ''"
                                        @click="goToPage(page)"
                                        x-text="page"></button>
                            </template>
                            
                            <button class="pagination-btn next-btn" 
                                    @click="nextPage" 
                                    :disabled="currentPage === totalPages">
                                <i class="bi bi-chevron-right"></i>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Filter Modal -->
    <template x-if="showFilterModal">
        <div class="modal fade show d-block"
             tabindex="-1"
             @click.self="showFilterModal = false"
             style="background-color: rgba(0,0,0,0.5);">
            <div class="modal-dialog modal-dialog-centered" style="max-width: 400px;">
                <div class="modal-content" style="border-radius: 16px; border: none;">
                    <div class="modal-header border-0 pb-2">
                        <h5 class="modal-title fw-bold d-flex align-items-center gap-2">
                            <span style="color: #7c3aed;">
                                <i class="bi bi-funnel-fill"></i>
                            </span>
                            Filter
                        </h5>
                        <button type="button" 
                                class="btn-close" 
                                @click="showFilterModal = false"></button>
                    </div>
                    <div class="modal-body px-4 py-3">
                        <form>
                            <!-- Location Filter -->
                            <div class="mb-4">
                                <label class="form-label fw-semibold mb-2" style="font-size: 14px; color: #666;">Location</label>
                                <input type="text" 
                                       class="form-control" 
                                       x-model="filters.location"
                                       placeholder="Enter location"
                                       style="border-radius: 10px; padding: 10px 15px; border: 1px solid #dee2e6; background-color: #f8f9fa;">
                            </div>

                            <!-- Bags Range -->
                            <div class="mb-4">
                                <label class="form-label fw-semibold mb-2" style="font-size: 14px; color: #666;">Collected Bags</label>
                                <select class="form-select" 
                                        x-model="filters.bagsRange"
                                        style="border-radius: 10px; padding: 10px 15px; border: 1px solid #dee2e6; background-color: #f8f9fa;">
                                    <option value="">All</option>
                                    <option value="0-5">0-5 bags</option>
                                    <option value="6-10">6-10 bags</option>
                                    <option value="10+">10+ bags</option>
                                </select>
                            </div>
                        </form>
                    </div>
                    <div class="modal-footer border-0 pt-0 px-4 pb-4">
                        <button type="button" 
                                class="btn btn-light" 
                                @click="resetFilters()"
                                style="border-radius: 10px; padding: 10px 28px; border: 1px solid #dee2e6; font-weight: 500;">
                            Cancel
                        </button>
                        <button type="button" 
                                class="btn text-white" 
                                @click="applyFilters()"
                                style="background-color: #4598F9; border-radius: 10px; padding: 10px 28px; font-weight: 500;">
                            Apply Filter
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </template>

    <!-- Add/Edit Collector Modal -->
    <template x-if="showModal">
        <div class="modal fade show d-block"
             tabindex="-1"
             @click.self="resetModal()"
             style="background-color: rgba(0,0,0,0.5);">
            <div class="modal-dialog modal-dialog-centered">
                <div class="modal-content" style="border-radius: 12px; border: none;">
                    <div class="modal-header border-0 pb-0">
                        <h5 class="modal-title fw-bold" x-text="isEditing ? 'Edit Collector' : 'Add Collector'"></h5>
                        <button type="button" class="btn-close" @click="resetModal()"></button>
                    </div>
                    <div class="modal-body px-4 py-4">
                        <form @submit.prevent="saveCollector()">
                            <div class="mb-3">
                                <label class="form-label text-muted" style="font-size: 14px;">Collector Name</label>
                                <input type="text" 
                                       class="form-control" 
                                       x-model="collector.name" 
                                       placeholder="Enter name"
                                       required>
                            </div>
                            <div class="mb-3">
                                <label class="form-label text-muted" style="font-size: 14px;">Location</label>
                                <input type="text" 
                                       class="form-control" 
                                       x-model="collector.location" 
                                       placeholder="Enter location"
                                       required>
                            </div>
                            <div class="mb-3">
                                <label class="form-label text-muted" style="font-size: 14px;">Phone Number</label>
                                <input type="text" 
                                       class="form-control" 
                                       x-model="collector.phoneNumber" 
                                       placeholder="+966681918737"
                                       required>
                            </div>
                            <div class="mb-3">
                                <label class="form-label text-muted" style="font-size: 14px;">Collected Bags</label>
                                <input type="number" 
                                       class="form-control" 
                                       x-model="collector.collectedBags" 
                                       placeholder="0"
                                       required>
                            </div>
                        </form>
                    </div>
                    <div class="modal-footer border-0 pt-0 px-4 pb-4">
                        <button type="button" 
                                class="btn btn-light"
                                @click="resetModal()"
                                style="border-radius: 8px; padding: 8px 24px; border: 1px solid #dee2e6;">
                            Cancel
                        </button>
                        <button type="button" 
                                class="btn text-white"
                                @click="saveCollector()"
                                style="background-color: #4598F9; border-radius: 8px; padding: 8px 24px;">
                            Save
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </template>
</div>

<script>
document.addEventListener('alpine:init', () => {
    Alpine.data('collectorsManager', () => ({
        showModal: false,
        showFilterModal: false,
        isEditing: false,
        editIndex: null,
        collector: {
            name: '',
            location: '',
            collectedBags: 0,
            phoneNumber: ''
        },
        searchQuery: '',
        sortOrder: 'asc',
        filters: {
            location: '',
            bagsRange: ''
        },
        currentPage: 1,
        itemsPerPage: 10,

        collectors: [
            { name: 'Sunnyvale Health Center', location: '123 Maple Street, Springfield, IL 62701', collectedBags: 5, phoneNumber: '+966655555555' },
            { name: 'Mountainview Medical Practice', location: '456 Oak Avenue, Rivertown, TX 75001', collectedBags: 10, phoneNumber: '+966677777777' },
            { name: 'Greenwood Family Clinic', location: '321 Birch Lane, Lakeview, NY 10001', collectedBags: 8, phoneNumber: '+966687876655' },
            { name: 'Riverbend Medical Group', location: '468 Chestnut Terrace, Forest Hill, PA 19001', collectedBags: 4, phoneNumber: '+966687654578' },
            { name: 'Maplewood Wellness Clinic', location: '654 Cedar Boulevard, Greenfield, FL 33101', collectedBags: 1, phoneNumber: '+966667452345' },
            { name: 'Oakridge Community Clinic', location: '135 Willow Way, Meadowbrook, OR 97001', collectedBags: 10, phoneNumber: '+966687453412' },
            { name: 'Lakeside Family Health', location: '357 Fir Street, Oceanview, MA 02101', collectedBags: 7, phoneNumber: '+966681918737' },
            { name: 'Pine Valley Health Clinic', location: '789 Pine Road, Hilltop, CA 90210', collectedBags: 3, phoneNumber: '+966688775566' },
            { name: 'Cedar Grove Health Services', location: '987 Elm Drive, Brookside, WA 98001', collectedBags: 3, phoneNumber: '+966662738293' },
            { name: 'Silver Lake Medical Center', location: '246 Spruce Circle, Valleyview, CO 80201', collectedBags: 10, phoneNumber: '+966698726398' }
        ],

        get filteredCollectors() {
            let list = this.collectors;
            
            // Search filter
            if (this.searchQuery.trim() !== '') {
                const query = this.searchQuery.toLowerCase();
                list = list.filter(item => 
                    item.name.toLowerCase().includes(query) || 
                    item.location.toLowerCase().includes(query) || 
                    item.phoneNumber.includes(query)
                );
            }
            
            // Location filter
            if (this.filters.location) {
                list = list.filter(item => 
                    item.location.toLowerCase().includes(this.filters.location.toLowerCase())
                );
            }
            
            // Bags range filter
            if (this.filters.bagsRange) {
                if (this.filters.bagsRange === '0-5') {
                    list = list.filter(item => item.collectedBags >= 0 && item.collectedBags <= 5);
                } else if (this.filters.bagsRange === '6-10') {
                    list = list.filter(item => item.collectedBags >= 6 && item.collectedBags <= 10);
                } else if (this.filters.bagsRange === '10+') {
                    list = list.filter(item => item.collectedBags > 10);
                }
            }
            
            // Sorting
            if (this.sortOrder === 'asc') {
                list = list.slice().sort((a, b) => a.name.localeCompare(b.name));
            } else {
                list = list.slice().sort((a, b) => b.name.localeCompare(a.name));
            }
            
            return list;
        },

        get totalPages() {
            return Math.ceil(this.filteredCollectors.length / this.itemsPerPage);
        },

        get startIndex() {
            return (this.currentPage - 1) * this.itemsPerPage;
        },

        get endIndex() {
            const end = this.startIndex + this.itemsPerPage;
            return end > this.filteredCollectors.length ? this.filteredCollectors.length : end;
        },

        get paginatedCollectors() {
            return this.filteredCollectors.slice(this.startIndex, this.endIndex);
        },

        get visiblePages() {
            const total = this.totalPages;
            const current = this.currentPage;
            const delta = 2;
            
            let range = [];
            let rangeWithDots = [];
            let l;

            for (let i = 1; i <= total; i++) {
                if (i == 1 || i == total || (i >= current - delta && i <= current + delta)) {
                    range.push(i);
                }
            }

            range.forEach((i) => {
                if (l) {
                    if (i - l === 2) {
                        rangeWithDots.push(l + 1);
                    } else if (i - l !== 1) {
                        rangeWithDots.push('...');
                    }
                }
                rangeWithDots.push(i);
                l = i;
            });

            return rangeWithDots;
        },

        applyFilters() {
            this.currentPage = 1;
            this.showFilterModal = false;
        },

        resetFilters() {
            this.filters = {
                location: '',
                bagsRange: ''
            };
            this.showFilterModal = false;
            this.currentPage = 1;
        },

        resetPagination() {
            this.currentPage = 1;
        },

        goToPage(page) {
            if (page !== '...' && page >= 1 && page <= this.totalPages) {
                this.currentPage = page;
            }
        },

        prevPage() {
            if (this.currentPage > 1) {
                this.currentPage--;
            }
        },

        nextPage() {
            if (this.currentPage < this.totalPages) {
                this.currentPage++;
            }
        },

        saveCollector() {
            if (!this.collector.name || !this.collector.location || !this.collector.phoneNumber) return;
            
            if (this.isEditing && this.editIndex !== null) {
                this.collectors[this.editIndex] = {...this.collector};
            } else {
                this.collectors.unshift({...this.collector});
            }
            this.resetModal();
        },

        openAddModal() {
            this.resetModal();
            this.showModal = true;
            this.isEditing = false;
        },

        openEditModal(index) {
            const item = this.paginatedCollectors[index];
            this.collector = {...item};
            this.editIndex = this.collectors.indexOf(item);
            this.isEditing = true;
            this.showModal = true;
        },

        deleteCollector(index) {
            if (confirm('Are you sure you want to delete this collector?')) {
                const item = this.paginatedCollectors[index];
                const originalIndex = this.collectors.indexOf(item);
                if (originalIndex > -1) this.collectors.splice(originalIndex, 1);
            }
        },

        viewBagHistory(index) {
            const item = this.paginatedCollectors[index];
            alert(`Viewing bag history for: ${item.name}\nTotal bags collected: ${item.collectedBags}`);
        },

        resetModal() {
            this.collector = {
                name: '',
                location: '',
                collectedBags: 0,
                phoneNumber: ''
            };
            this.editIndex = null;
            this.isEditing = false;
            this.showModal = false;
        },

        toggleSort() {
            this.sortOrder = this.sortOrder === 'asc' ? 'desc' : 'asc';
        },

        exportCSV() {
            alert('Exporting CSV file...');
            // CSV export functionality would go here
        }
    }));
});
</script>

<style>
    [x-cloak] { display: none !important; }
    .modal { z-index: 1050; }
    
    .pagination-container {
        display: flex;
        align-items: center;
        gap: 5px;
    }
    
    .pagination-btn {
        width: 36px;
        height: 36px;
        border-radius: 8px;
        border: 1px solid #e0e0e0;
        background-color: #fff;
        display: flex;
        align-items: center;
        justify-content: center;
        font-weight: 500;
        color: #333;
        transition: all 0.2s;
    }
    
    .pagination-btn:hover:not(:disabled) {
        background-color: #f5f5f5;
    }
    
    .pagination-btn.active {
        background-color: #4598F9;
        color: white;
        border-color: #4598F9;
    }
    
    .pagination-btn:disabled {
        opacity: 0.5;
        cursor: not-allowed;
    }
    
    .pagination-btn.prev-btn, 
    .pagination-btn.next-btn {
        font-size: 14px;
    } 
    .table thead th:first-child {
        border: 10px solid green !important;
        background-color: red !important;
    }       
</style>

@endsection
