<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\WasteType;
use App\Models\Clinic;
use Illuminate\Support\Facades\DB;

class WasteTypeController extends Controller
{
    /**
     * Display a listing of waste types for a specific clinic
     */
    public function index($clinic_id)
    {
        try {
            $clinic = Clinic::findOrFail($clinic_id);
            $wasteTypes = WasteType::where('clinic_id', $clinic_id)
                ->orderBy('created_at', 'desc')
                ->get();

            return view('Waste-type', compact('wasteTypes', 'clinic'));
        } catch (\Exception $e) {
            return redirect()->route('moduls', ['clinic_id' => $clinic_id])
                ->with('error', 'Clinic not found');
        }
    }

    /**
     * Store a newly created waste type
     */
    public function store(Request $request, $clinic_id)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:150',
            ]);

            $wasteType = WasteType::create([
                'clinic_id' => $clinic_id,
                'name' => $validated['name'],
                'status' => 1,
            ]);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Waste type created successfully!',
                    'wasteType' => $wasteType
                ]);
            }

            return redirect()->route('waste-type.index', ['clinic_id' => $clinic_id])
                ->with('success', 'Waste type created successfully!');

        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => array_values($e->errors())[0][0] ?? 'Validation error',
                    'errors' => $e->errors()
                ], 422);
            }

            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'An error occurred while creating waste type: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Update the specified waste type
     */
    public function update(Request $request, $clinic_id, $id)
    {
        try {
            $wasteType = WasteType::where('clinic_id', $clinic_id)->findOrFail($id);

            $validated = $request->validate([
                'name' => 'required|string|max:150',
            ]);

            $wasteType->update([
                'name' => $validated['name'],
            ]);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Waste type updated successfully!',
                    'wasteType' => $wasteType
                ]);
            }

            return redirect()->route('waste-type.index', ['clinic_id' => $clinic_id])
                ->with('success', 'Waste type updated successfully!');

        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => array_values($e->errors())[0][0] ?? 'Validation error',
                    'errors' => $e->errors()
                ], 422);
            }

            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'An error occurred while updating waste type: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified waste type
     */
    public function destroy($clinic_id, $id)
    {
        try {
            $wasteType = WasteType::where('clinic_id', $clinic_id)->findOrFail($id);
            $wasteType->delete();

            if (request()->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Waste type deleted successfully!'
                ]);
            }

            return redirect()->route('waste-type.index', ['clinic_id' => $clinic_id])
                ->with('success', 'Waste type deleted successfully!');

        } catch (\Exception $e) {
            if (request()->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'An error occurred while deleting waste type: ' . $e->getMessage());
        }
    }

    /**
     * Get waste type details for viewing
     */
    public function show($clinic_id, $id)
    {
        try {
            $wasteType = WasteType::where('clinic_id', $clinic_id)->findOrFail($id);

            if (request()->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'wasteType' => $wasteType
                ]);
            }

            return response()->json([
                'success' => true,
                'wasteType' => $wasteType
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Waste type not found'
            ], 404);
        }
    }
}
