<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\WasteBag;
use App\Models\Clinic;
use App\Models\WasteType;
use App\Models\ContainerType;
use App\Models\HospitalDepartment;
use App\Models\Collector;

class WasteBagController extends Controller
{
    /**
     * Display a listing of waste bags for a specific clinic
     */
    public function index($clinic_id)
    {
        try {
            $clinic = Clinic::findOrFail($clinic_id);

            // Get waste bags with relationships, ordered by created_timestamp descending
            $wasteBags = WasteBag::with(['wasteType', 'containerType', 'department', 'collector'])
                ->where('clinic_id', $clinic_id)
                ->orderBy('created_timestamp', 'desc')
                ->paginate(10);

            // Get related data for dropdowns
            $wasteTypes = WasteType::where('clinic_id', $clinic_id)->get();
            $containerTypes = ContainerType::where('clinic_id', $clinic_id)->get();
            $departments = HospitalDepartment::all();
            $collectors = Collector::where('clinic_id', $clinic_id)->get();

            return view('Waste-bags', compact('clinic', 'wasteBags', 'wasteTypes', 'containerTypes', 'departments', 'collectors'));
        } catch (\Exception $e) {
            return redirect()->route('Medical-Waste', ['clinic_id' => $clinic_id])
                ->with('error', 'Clinic not found');
        }
    }

    /**
     * Store a newly created waste bag
     */
    public function store(Request $request, $clinic_id)
    {
        try {
            $validated = $request->validate([
                'waste_type_id' => 'required|integer|exists:waste_types,id',
                'container_type_id' => 'nullable|integer|exists:container_type,id',
                'department_id' => 'nullable|integer|exists:hospital_departments,id',
                'collector_id' => 'nullable|integer|exists:collector,id',
                'status' => 'required|integer|in:1,2,3',
                'created_by_id' => 'nullable|integer',
                'closed_by_id' => 'nullable|integer',
                'delivered_by_id' => 'nullable|integer',
            ]);

            $wasteBag = WasteBag::create([
                'clinic_id' => $clinic_id,
                'waste_type_id' => $validated['waste_type_id'],
                'container_type_id' => $validated['container_type_id'] ?? null,
                'department_id' => $validated['department_id'] ?? null,
                'collector_id' => $validated['collector_id'] ?? null,
                'status' => $validated['status'],
                'created_by_id' => $validated['created_by_id'] ?? null,
                'closed_by_id' => $validated['closed_by_id'] ?? null,
                'delivered_by_id' => $validated['delivered_by_id'] ?? null,
                'created_timestamp' => now(),
                'closed_timestamp' => $validated['status'] == WasteBag::STATUS_CLOSED ? now() : null,
                'delivered_timestamp' => $validated['status'] == WasteBag::STATUS_DELIVERED ? now() : null,
            ]);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Waste bag created successfully!',
                    'wasteBag' => $wasteBag->load(['wasteType', 'containerType', 'department', 'collector'])
                ]);
            }

            return redirect()->route('waste-bags.index', ['clinic_id' => $clinic_id])
                ->with('success', 'Waste bag created successfully!');

        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => array_values($e->errors())[0][0] ?? 'Validation error',
                    'errors' => $e->errors()
                ], 422);
            }

            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'An error occurred while creating waste bag: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Update the specified waste bag
     */
    public function update(Request $request, $clinic_id, $id)
    {
        try {
            $wasteBag = WasteBag::where('clinic_id', $clinic_id)->findOrFail($id);

            $validated = $request->validate([
                'waste_type_id' => 'required|integer|exists:waste_types,id',
                'container_type_id' => 'nullable|integer|exists:container_type,id',
                'department_id' => 'nullable|integer|exists:hospital_departments,id',
                'collector_id' => 'nullable|integer|exists:collector,id',
                'status' => 'required|integer|in:1,2,3',
                'created_by_id' => 'nullable|integer',
                'closed_by_id' => 'nullable|integer',
                'delivered_by_id' => 'nullable|integer',
            ]);

            $updateData = [
                'waste_type_id' => $validated['waste_type_id'],
                'container_type_id' => $validated['container_type_id'] ?? null,
                'department_id' => $validated['department_id'] ?? null,
                'collector_id' => $validated['collector_id'] ?? null,
                'status' => $validated['status'],
                'created_by_id' => $validated['created_by_id'] ?? null,
                'closed_by_id' => $validated['closed_by_id'] ?? null,
                'delivered_by_id' => $validated['delivered_by_id'] ?? null,
            ];

            // Update timestamps based on status changes
            if ($validated['status'] == WasteBag::STATUS_CLOSED && !$wasteBag->closed_timestamp) {
                $updateData['closed_timestamp'] = now();
            }
            if ($validated['status'] == WasteBag::STATUS_DELIVERED && !$wasteBag->delivered_timestamp) {
                $updateData['delivered_timestamp'] = now();
            }

            $wasteBag->update($updateData);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Waste bag updated successfully!',
                    'wasteBag' => $wasteBag->load(['wasteType', 'containerType', 'department', 'collector'])
                ]);
            }

            return redirect()->route('waste-bags.index', ['clinic_id' => $clinic_id])
                ->with('success', 'Waste bag updated successfully!');

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'An error occurred while updating waste bag: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified waste bag
     */
    public function destroy($clinic_id, $id)
    {
        try {
            $wasteBag = WasteBag::where('clinic_id', $clinic_id)->findOrFail($id);
            $wasteBag->delete();

            if (request()->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Waste bag deleted successfully!'
                ]);
            }

            return redirect()->route('waste-bags.index', ['clinic_id' => $clinic_id])
                ->with('success', 'Waste bag deleted successfully!');

        } catch (\Exception $e) {
            if (request()->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'An error occurred while deleting waste bag: ' . $e->getMessage());
        }
    }

    /**
     * Get waste bag details for viewing
     */
    public function show($clinic_id, $id)
    {
        try {
            $wasteBag = WasteBag::with(['wasteType', 'containerType', 'department', 'collector'])
                ->where('clinic_id', $clinic_id)
                ->findOrFail($id);

            if (request()->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'wasteBag' => $wasteBag
                ]);
            }

            return response()->json([
                'success' => true,
                'wasteBag' => $wasteBag
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Waste bag not found'
            ], 404);
        }
    }

    /**
     * Update waste bag status
     */
    public function updateStatus(Request $request, $clinic_id, $id)
    {
        try {
            $wasteBag = WasteBag::where('clinic_id', $clinic_id)->findOrFail($id);

            $validated = $request->validate([
                'status' => 'required|integer|in:1,2,3',
            ]);

            $newStatus = $validated['status'];
            $updateData = ['status' => $newStatus];

            // Update timestamps based on status changes
            if ($newStatus == WasteBag::STATUS_CLOSED && !$wasteBag->closed_timestamp) {
                $updateData['closed_timestamp'] = now();
            }
            if ($newStatus == WasteBag::STATUS_DELIVERED && !$wasteBag->delivered_timestamp) {
                $updateData['delivered_timestamp'] = now();
            }

            $wasteBag->update($updateData);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Status updated successfully!',
                    'wasteBag' => $wasteBag->fresh()
                ]);
            }

            return redirect()->route('Waste-bags', ['clinic_id' => $clinic_id])
                ->with('success', 'Status updated successfully!');

        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'An error occurred while updating status: ' . $e->getMessage());
        }
    }
}
