<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Room;

class RoomController extends Controller
{
    /**
     * Display room management page (accessed from add clinic form)
     */
    public function index()
    {
        // Get all master rooms (clinic_id = NULL)
        $rooms = Room::whereNull('clinic_id')
            ->orderBy('name', 'asc')
            ->get();

        return view('Room', compact('rooms'));
    }

    /**
     * Store a new master room
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:150',
            ]);

            // Check if room already exists in master list
            $existingRoom = Room::whereNull('clinic_id')
                ->where('name', $validated['name'])
                ->first();

            if ($existingRoom) {
                return response()->json([
                    'success' => false,
                    'message' => 'This room already exists in the master list.'
                ], 422);
            }

            $room = Room::create([
                'clinic_id' => null, // NULL means master room
                'name' => $validated['name'],
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Room added successfully!',
                'room' => $room
            ], 201);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => $e->errors()['name'][0] ?? 'Validation error'
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update a master room
     */
    public function update(Request $request, $id)
    {
        try {
            $room = Room::where('id', $id)
                ->whereNull('clinic_id')
                ->firstOrFail();

            $validated = $request->validate([
                'name' => 'required|string|max:150',
            ]);

            // Check if another room with this name exists in master list
            $existingRoom = Room::whereNull('clinic_id')
                ->where('name', $validated['name'])
                ->where('id', '!=', $id)
                ->first();

            if ($existingRoom) {
                return response()->json([
                    'success' => false,
                    'message' => 'This room name already exists.'
                ], 422);
            }

            $room->update([
                'name' => $validated['name'],
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Room updated successfully!',
                'room' => $room
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete a master room
     */
    public function destroy($id)
    {
        try {
            $room = Room::where('id', $id)
                ->whereNull('clinic_id')
                ->firstOrFail();

            $room->delete();

            return response()->json([
                'success' => true,
                'message' => 'Room deleted successfully!'
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 404);
        }
    }
}
