<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\ResponsiblePerson;
use Illuminate\Support\Facades\Hash;

class ResponsiblePersonController extends Controller
{
    /**
     * Display responsible persons management page
     */
    public function index()
    {
        // Get all master responsible persons (clinic_id = NULL)
        $persons = ResponsiblePerson::whereNull('clinic_id')
            ->orderBy('name', 'asc')
            ->get();

        return view('All-persons', compact('persons'));
    }

    /**
     * Store a new master responsible person
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:100',
                'username' => 'required|string|max:100|unique:responsible_person,username',
                'password' => 'required|string|min:6',
                'role' => 'required|in:Admin,Manager,User',
                'access_level' => 'nullable|array',
                'access_level.*' => 'nullable|string',
            ], [
                'username.unique' => 'This username already exists.',
            ]);

            // If role is Admin, give all access
            $accessLevel = $validated['role'] === 'Admin'
                ? ['cleaning', 'disinfection', 'sterilization', 'inventory', 'task', 'medical_waste']
                : ($validated['access_level'] ?? []);

            $person = ResponsiblePerson::create([
                'clinic_id' => null, // NULL means master person
                'name' => $validated['name'],
                'username' => $validated['username'],
                'password' => Hash::make($validated['password']),
                'role' => $validated['role'],
                'access_level' => json_encode($accessLevel),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Responsible person added successfully!',
                'person' => $person
            ], 201);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => array_values($e->errors())[0][0] ?? 'Validation error',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update a master responsible person
     */
    public function update(Request $request, $id)
    {
        try {
            $person = ResponsiblePerson::where('id', $id)
                ->whereNull('clinic_id')
                ->firstOrFail();

            $validated = $request->validate([
                'name' => 'required|string|max:100',
                'username' => 'required|string|max:100|unique:responsible_person,username,' . $id,
                'password' => 'nullable|string|min:6',
                'role' => 'required|in:Admin,Manager,User',
                'access_level' => 'nullable|array',
                'access_level.*' => 'nullable|string',
            ]);

            // If role is Admin, give all access
            $accessLevel = $validated['role'] === 'Admin'
                ? ['cleaning', 'disinfection', 'sterilization', 'inventory', 'task', 'medical_waste']
                : ($validated['access_level'] ?? []);

            $updateData = [
                'name' => $validated['name'],
                'username' => $validated['username'],
                'role' => $validated['role'],
                'access_level' => json_encode($accessLevel),
            ];

            // Only update password if provided
            if (!empty($validated['password'])) {
                $updateData['password'] = Hash::make($validated['password']);
            }

            $person->update($updateData);

            return response()->json([
                'success' => true,
                'message' => 'Responsible person updated successfully!',
                'person' => $person
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete a master responsible person
     */
    public function destroy($id)
    {
        try {
            $person = ResponsiblePerson::where('id', $id)
                ->whereNull('clinic_id')
                ->firstOrFail();

            $person->delete();

            return response()->json([
                'success' => true,
                'message' => 'Responsible person deleted successfully!'
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 404);
        }
    }
}
