<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Invoice;
use App\Models\Clinic;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use ZipArchive;

class InvoiceController extends Controller
{
    public function index()
    {
        // Check if user has Super Admin access
        $role = session('admin_role');
        if ($role !== 'Super Admin') {
            abort(403, 'Access denied. Only Super Admin can access this section.');
        }

        $invoices = Invoice::with('clinic')
            ->orderBy('invoice_date', 'desc')
            ->get();

        // Format invoices for frontend
        $formattedInvoices = $invoices->map(function($invoice) {
            return [
                'id' => $invoice->id,
                'name' => $invoice->clinic->name ?? $invoice->clinic->clinic_name,
                'location' => $invoice->clinic->location ?? $invoice->clinic->clinic_location,
                'paid_date' => $invoice->paid_date ?? $invoice->invoice_date->format('Y-m-d'),
                'status' => ucfirst($invoice->status),
                'invoice_number' => $invoice->invoice_number,
                'amount' => number_format($invoice->amount, 2),
                'invoice_date' => $invoice->invoice_date->format('Y-m-d')
            ];
        })->values()->toArray();

        return view('payment', compact('formattedInvoices'));
    }

    public function generateMonthlyInvoices()
    {
        $currentDate = Carbon::now();

        // Only generate on the 15th of the month
        if ($currentDate->day != 15) {
            return response()->json(['message' => 'Invoices are only generated on the 15th of each month'], 400);
        }

        $clinics = Clinic::where('status', 1)->get();
        $generatedInvoices = [];

        foreach ($clinics as $clinic) {
            // Check if invoice already exists for this month
            $existingInvoice = Invoice::where('clinic_id', $clinic->id)
                ->whereYear('invoice_date', $currentDate->year)
                ->whereMonth('invoice_date', $currentDate->month)
                ->first();

            if (!$existingInvoice) {
                $invoice = Invoice::create([
                    'clinic_id' => $clinic->id,
                    'invoice_number' => $this->generateInvoiceNumber(),
                    'invoice_date' => $currentDate->toDateString(),
                    'amount' => $this->calculateInvoiceAmount($clinic->id),
                    'status' => 'pending'
                ]);

                // Generate PDF
                $pdfPath = $this->generateInvoicePDF($invoice);
                $invoice->update(['pdf_path' => $pdfPath]);

                $generatedInvoices[] = $invoice;
            }
        }

        return response()->json([
            'message' => 'Monthly invoices generated successfully',
            'count' => count($generatedInvoices)
        ]);
    }

    public function downloadInvoice($id)
    {
        $invoice = Invoice::with('clinic')->findOrFail($id);

        if (!$invoice->pdf_path || !Storage::exists($invoice->pdf_path)) {
            $pdfPath = $this->generateInvoicePDF($invoice);
            $invoice->update(['pdf_path' => $pdfPath]);
        }

        return Storage::download($invoice->pdf_path, "Invoice-{$invoice->invoice_number}.pdf");
    }

    public function downloadAllInvoices(Request $request)
    {
        $month = $request->input('month');
        $year = $request->input('year');

        $query = Invoice::with('clinic');

        if ($month && $year) {
            $query->whereYear('invoice_date', $year)
                  ->whereMonth('invoice_date', $month);
        }

        $invoices = $query->get();

        if ($invoices->isEmpty()) {
            return response()->json(['message' => 'No invoices found'], 404);
        }

        // Create ZIP file
        $zipFileName = 'invoices_' . ($month ? "{$year}_{$month}" : 'all') . '.zip';
        $zipPath = storage_path('app/temp/' . $zipFileName);

        // Ensure temp directory exists
        if (!file_exists(storage_path('app/temp'))) {
            mkdir(storage_path('app/temp'), 0755, true);
        }

        $zip = new ZipArchive;
        if ($zip->open($zipPath, ZipArchive::CREATE | ZipArchive::OVERWRITE) === TRUE) {
            foreach ($invoices as $invoice) {
                if (!$invoice->pdf_path || !Storage::exists($invoice->pdf_path)) {
                    $pdfPath = $this->generateInvoicePDF($invoice);
                    $invoice->update(['pdf_path' => $pdfPath]);
                }

                $zip->addFile(
                    storage_path('app/' . $invoice->pdf_path),
                    "Invoice-{$invoice->invoice_number}.pdf"
                );
            }
            $zip->close();
        }

        return response()->download($zipPath)->deleteFileAfterSend(true);
    }

    private function generateInvoiceNumber()
    {
        $prefix = 'INV';
        $year = date('Y');
        $month = date('m');

        $lastInvoice = Invoice::whereYear('invoice_date', $year)
            ->whereMonth('invoice_date', $month)
            ->orderBy('id', 'desc')
            ->first();

        $number = $lastInvoice ? (intval(substr($lastInvoice->invoice_number, -4)) + 1) : 1;

        return sprintf('%s-%s%s-%04d', $prefix, $year, $month, $number);
    }

    private function calculateInvoiceAmount($clinicId)
    {
        // Calculate based on waste bags or fixed monthly fee
        // For now, using a fixed amount of $500 per month
        // You can customize this based on your business logic
        return 500.00;
    }

    private function generateInvoicePDF($invoice)
    {
        $invoice->load('clinic');

        $pdf = Pdf::loadView('invoices.invoice-pdf', compact('invoice'));

        $fileName = "invoice_{$invoice->invoice_number}.pdf";
        $path = "invoices/{$fileName}";

        Storage::put($path, $pdf->output());

        return $path;
    }

    public function updateStatus(Request $request, $id)
    {
        $invoice = Invoice::findOrFail($id);

        $request->validate([
            'status' => 'required|in:pending,paid,overdue',
            'paid_date' => 'required_if:status,paid|date|nullable'
        ]);

        $invoice->update([
            'status' => $request->status,
            'paid_date' => $request->status === 'paid' ? $request->paid_date : null
        ]);

        return response()->json(['message' => 'Invoice status updated successfully']);
    }
}
