<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\HospitalDepartment;

class HospitalDepartmentController extends Controller
{
    /**
     * Display hospital department management page
     */
    public function index()
    {
        // Get all master departments (clinic_id = NULL)
        $departments = HospitalDepartment::whereNull('clinic_id')
            ->orderBy('name', 'asc')
            ->get();

        return view('hospital-department', compact('departments'));
    }

    /**
     * Store a new master department
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:150',
            ]);

            // Check if department already exists in master list
            $existingDept = HospitalDepartment::whereNull('clinic_id')
                ->where('name', $validated['name'])
                ->first();

            if ($existingDept) {
                return response()->json([
                    'success' => false,
                    'message' => 'This department already exists in the master list.'
                ], 422);
            }

            $department = HospitalDepartment::create([
                'clinic_id' => null, // NULL means master department
                'name' => $validated['name'],
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Department added successfully!',
                'department' => $department
            ], 201);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => $e->errors()['name'][0] ?? 'Validation error'
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update a master department
     */
    public function update(Request $request, $id)
    {
        try {
            $department = HospitalDepartment::where('id', $id)
                ->whereNull('clinic_id')
                ->firstOrFail();

            $validated = $request->validate([
                'name' => 'required|string|max:150',
            ]);

            // Check if another department with this name exists in master list
            $existingDept = HospitalDepartment::whereNull('clinic_id')
                ->where('name', $validated['name'])
                ->where('id', '!=', $id)
                ->first();

            if ($existingDept) {
                return response()->json([
                    'success' => false,
                    'message' => 'This department name already exists.'
                ], 422);
            }

            $department->update([
                'name' => $validated['name'],
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Department updated successfully!',
                'department' => $department
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete a master department
     */
    public function destroy($id)
    {
        try {
            $department = HospitalDepartment::where('id', $id)
                ->whereNull('clinic_id')
                ->firstOrFail();

            $department->delete();

            return response()->json([
                'success' => true,
                'message' => 'Department deleted successfully!'
            ], 200);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 404);
        }
    }
}
