<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Document;
use Illuminate\Support\Facades\Storage;

class DocumentController extends Controller
{
    /**
     * Get all documents
     */
    public function index()
    {
        $documents = Document::orderBy('created_at', 'desc')->get();
        return response()->json($documents);
    }

    /**
     * Store a new document
     */
    public function store(Request $request)
    {
        try {
            // Check if file was actually uploaded (catches PHP upload_max_filesize issues)
            if (!$request->hasFile('document') && $request->has('name')) {
                return response()->json([
                    'message' => 'File upload failed. The file may be too large for server configuration. Current PHP upload limit is ' . ini_get('upload_max_filesize') . '. Please choose a smaller file or contact administrator.'
                ], 413);
            }

            $validated = $request->validate([
                'name' => 'required|string|max:150',
                'document' => 'required|file|mimes:pdf,doc,docx,xls,xlsx,jpg,jpeg,png|max:10240', // Max 10MB
            ]);

            // Handle file upload
            if ($request->hasFile('document')) {
                $file = $request->file('document');

                // Check if file is valid
                if (!$file->isValid()) {
                    return response()->json([
                        'message' => 'File upload failed: ' . $file->getErrorMessage()
                    ], 400);
                }

                $fileName = time() . '_' . $file->getClientOriginalName();
                $filePath = $file->storeAs('documents', $fileName, 'public');

                // Create document record
                $document = Document::create([
                    'name' => $validated['name'],
                    'document' => $filePath,
                ]);

                return response()->json([
                    'message' => 'Document uploaded successfully!',
                    'document' => $document
                ], 201);
            }

            return response()->json([
                'message' => 'No file uploaded.'
            ], 400);
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Return detailed validation errors
            $errors = $e->errors();
            $errorMessages = [];

            foreach ($errors as $field => $messages) {
                $errorMessages[] = implode(' ', $messages);
            }

            return response()->json([
                'message' => implode(' ', $errorMessages)
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'message' => 'An error occurred while uploading the document: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete a document
     */
    public function destroy($id)
    {
        $document = Document::find($id);

        if (!$document) {
            return response()->json([
                'message' => 'Document not found.'
            ], 404);
        }

        // Delete file from storage
        if (Storage::disk('public')->exists($document->document)) {
            Storage::disk('public')->delete($document->document);
        }

        $document->delete();

        return response()->json([
            'message' => 'Document deleted successfully!'
        ], 200);
    }

    /**
     * View a document (opens in browser)
     */
    public function view($id)
    {
        $document = Document::find($id);

        if (!$document) {
            return response()->json([
                'message' => 'Document not found.'
            ], 404);
        }

        $filePath = storage_path('app/public/' . $document->document);

        if (!file_exists($filePath)) {
            return response()->json([
                'message' => 'File not found.'
            ], 404);
        }

        // Return file to view in browser (inline)
        return response()->file($filePath);
    }

    /**
     * Download a document
     */
    public function download($id)
    {
        $document = Document::find($id);

        if (!$document) {
            return response()->json([
                'message' => 'Document not found.'
            ], 404);
        }

        $filePath = storage_path('app/public/' . $document->document);

        if (!file_exists($filePath)) {
            return response()->json([
                'message' => 'File not found.'
            ], 404);
        }

        // Get original filename from the path
        $originalFileName = basename($filePath);

        // Force download with original filename
        return response()->download($filePath, $originalFileName);
    }
}
