<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\ContainerType;
use App\Models\Clinic;

class ContainerTypeController extends Controller
{
    /**
     * Display a listing of container types for a specific clinic
     */
    public function index($clinic_id)
    {
        try {
            $clinic = Clinic::findOrFail($clinic_id);
            $containerTypes = ContainerType::where('clinic_id', $clinic_id)
                ->orderBy('created_at', 'desc')
                ->get();

            return view('container-type', compact('containerTypes', 'clinic'));
        } catch (\Exception $e) {
            return redirect()->route('moduls', ['clinic_id' => $clinic_id])
                ->with('error', 'Clinic not found');
        }
    }

    /**
     * Store a newly created container type
     */
    public function store(Request $request, $clinic_id)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:150',
            ]);

            $containerType = ContainerType::create([
                'clinic_id' => $clinic_id,
                'name' => $validated['name'],
            ]);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Container type created successfully!',
                    'containerType' => $containerType
                ]);
            }

            return redirect()->route('container-type.index', ['clinic_id' => $clinic_id])
                ->with('success', 'Container type created successfully!');

        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => array_values($e->errors())[0][0] ?? 'Validation error',
                    'errors' => $e->errors()
                ], 422);
            }

            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'An error occurred while creating container type: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Update the specified container type
     */
    public function update(Request $request, $clinic_id, $id)
    {
        try {
            $containerType = ContainerType::where('clinic_id', $clinic_id)->findOrFail($id);

            $validated = $request->validate([
                'name' => 'required|string|max:150',
            ]);

            $containerType->update([
                'name' => $validated['name'],
            ]);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Container type updated successfully!',
                    'containerType' => $containerType
                ]);
            }

            return redirect()->route('container-type.index', ['clinic_id' => $clinic_id])
                ->with('success', 'Container type updated successfully!');

        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => array_values($e->errors())[0][0] ?? 'Validation error',
                    'errors' => $e->errors()
                ], 422);
            }

            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'An error occurred while updating container type: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified container type
     */
    public function destroy($clinic_id, $id)
    {
        try {
            $containerType = ContainerType::where('clinic_id', $clinic_id)->findOrFail($id);
            $containerType->delete();

            if (request()->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Container type deleted successfully!'
                ]);
            }

            return redirect()->route('container-type.index', ['clinic_id' => $clinic_id])
                ->with('success', 'Container type deleted successfully!');

        } catch (\Exception $e) {
            if (request()->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'An error occurred while deleting container type: ' . $e->getMessage());
        }
    }

    /**
     * Get container type details for viewing
     */
    public function show($clinic_id, $id)
    {
        try {
            $containerType = ContainerType::where('clinic_id', $clinic_id)->findOrFail($id);

            if (request()->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'containerType' => $containerType
                ]);
            }

            return response()->json([
                'success' => true,
                'containerType' => $containerType
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Container type not found'
            ], 404);
        }
    }
}
