<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Collector;
use App\Models\Clinic;

class CollectorController extends Controller
{
    /**
     * Display a listing of collectors for a specific clinic
     */
    public function index($clinic_id)
    {
        try {
            $clinic = Clinic::findOrFail($clinic_id);
            $collectors = Collector::where('clinic_id', $clinic_id)
                ->orderBy('created_at', 'desc')
                ->get();

            return view('collectors', compact('collectors', 'clinic'));
        } catch (\Exception $e) {
            return redirect()->route('moduls', ['clinic_id' => $clinic_id])
                ->with('error', 'Clinic not found');
        }
    }

    /**
     * Store a newly created collector
     */
    public function store(Request $request, $clinic_id)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:150',
                'location' => 'nullable|string|max:255',
                'phone_number' => 'nullable|string|max:20',
            ]);

            $collector = Collector::create([
                'clinic_id' => $clinic_id,
                'name' => $validated['name'],
                'location' => $validated['location'] ?? null,
                'phone_number' => $validated['phone_number'] ?? null,
            ]);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Collector created successfully!',
                    'collector' => $collector
                ]);
            }

            return redirect()->route('collectors.index', ['clinic_id' => $clinic_id])
                ->with('success', 'Collector created successfully!');

        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => array_values($e->errors())[0][0] ?? 'Validation error',
                    'errors' => $e->errors()
                ], 422);
            }

            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'An error occurred while creating collector: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Update the specified collector
     */
    public function update(Request $request, $clinic_id, $id)
    {
        try {
            $collector = Collector::where('clinic_id', $clinic_id)->findOrFail($id);

            $validated = $request->validate([
                'name' => 'required|string|max:150',
                'location' => 'nullable|string|max:255',
                'phone_number' => 'nullable|string|max:20',
            ]);

            $collector->update([
                'name' => $validated['name'],
                'location' => $validated['location'] ?? null,
                'phone_number' => $validated['phone_number'] ?? null,
            ]);

            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Collector updated successfully!',
                    'collector' => $collector
                ]);
            }

            return redirect()->route('collectors.index', ['clinic_id' => $clinic_id])
                ->with('success', 'Collector updated successfully!');

        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => array_values($e->errors())[0][0] ?? 'Validation error',
                    'errors' => $e->errors()
                ], 422);
            }

            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'An error occurred while updating collector: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Remove the specified collector
     */
    public function destroy($clinic_id, $id)
    {
        try {
            $collector = Collector::where('clinic_id', $clinic_id)->findOrFail($id);
            $collector->delete();

            if (request()->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Collector deleted successfully!'
                ]);
            }

            return redirect()->route('collectors.index', ['clinic_id' => $clinic_id])
                ->with('success', 'Collector deleted successfully!');

        } catch (\Exception $e) {
            if (request()->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'An error occurred while deleting collector: ' . $e->getMessage());
        }
    }

    /**
     * Get collector details for viewing
     */
    public function show($clinic_id, $id)
    {
        try {
            $collector = Collector::where('clinic_id', $clinic_id)->findOrFail($id);

            if (request()->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'collector' => $collector
                ]);
            }

            return response()->json([
                'success' => true,
                'collector' => $collector
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Collector not found'
            ], 404);
        }
    }
}
