<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Clinic;
use App\Models\HospitalDepartment;
use App\Models\Room;
use App\Models\ResponsiblePerson;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;

class ClinicController extends Controller
{
    /**
     * Display a listing of clinics
     */
    public function index()
    {
        $clinics = Clinic::orderBy('created_at', 'desc')->get();
        return view('clinics', compact('clinics'));
    }

    /**
     * Show the form for creating a new clinic
     */
    public function create()
    {
        // Get all master data (clinic_id = NULL)
        $masterRooms = Room::whereNull('clinic_id')->orderBy('name', 'asc')->get();
        $masterDepartments = HospitalDepartment::whereNull('clinic_id')->orderBy('name', 'asc')->get();
        $masterPersons = ResponsiblePerson::whereNull('clinic_id')->orderBy('name', 'asc')->get();

        return view('addnewClinic', compact('masterRooms', 'masterDepartments', 'masterPersons'));
    }

    /**
     * Store a newly created clinic in database
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                // Clinic Details
                'clinic_name' => 'required|string|max:150',
                'clinic_location' => 'required|string|max:255',

                // Rooms, Departments, and Persons (JSON strings from localStorage)
                'rooms' => 'nullable|string',
                'departments' => 'nullable|string',
                'responsible_persons' => 'nullable|string',

                // Clinic Admin
                'admin_name' => 'nullable|string|max:100',
                'username' => 'nullable|string|max:100|unique:clinic,admin_username',
                'password' => 'nullable|string|min:6',

                // Module Access Level
                'clinic_modules' => 'nullable|array',
                'clinic_modules.*' => 'nullable|string',
            ]);

            // Start database transaction
            DB::beginTransaction();

            // Create clinic record
            $clinic = Clinic::create([
                'clinic_name' => $validated['clinic_name'],
                'clinic_location' => $validated['clinic_location'],
                'admin_name' => $validated['admin_name'] ?? null,
                'admin_username' => $validated['username'] ?? null,
                'admin_password' => isset($validated['password']) ? Hash::make($validated['password']) : null,
                'name' => $validated['clinic_name'], // Duplicate for compatibility
                'location' => $validated['clinic_location'], // Duplicate for compatibility
                'modul_access' => json_encode($validated['clinic_modules'] ?? []),
                'status' => 1, // Active by default
            ]);

            // Process rooms from JSON
            if (!empty($validated['rooms'])) {
                $rooms = json_decode($validated['rooms'], true);
                if (is_array($rooms)) {
                    foreach ($rooms as $room) {
                        if (!empty($room['name'])) {
                            Room::create([
                                'clinic_id' => $clinic->id,
                                'name' => $room['name'],
                            ]);
                        }
                    }
                }
            }

            // Process departments from JSON
            if (!empty($validated['departments'])) {
                $departments = json_decode($validated['departments'], true);
                if (is_array($departments)) {
                    foreach ($departments as $dept) {
                        if (!empty($dept['name'])) {
                            HospitalDepartment::create([
                                'clinic_id' => $clinic->id,
                                'name' => $dept['name'],
                            ]);
                        }
                    }
                }
            }

            // Process responsible persons from JSON
            if (!empty($validated['responsible_persons'])) {
                $persons = json_decode($validated['responsible_persons'], true);
                if (is_array($persons)) {
                    foreach ($persons as $person) {
                        if (!empty($person['name']) && !empty($person['username'])) {
                            ResponsiblePerson::create([
                                'clinic_id' => $clinic->id,
                                'name' => $person['name'],
                                'username' => $person['username'],
                                'password' => Hash::make($person['password']),
                                'role' => $person['role'],
                                'access_level' => json_encode($person['access_level'] ?? []),
                            ]);
                        }
                    }
                }
            }

            // Commit transaction
            DB::commit();

            return redirect()->route('clinics')
                ->with('success', 'Clinic created successfully!');

        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with('error', 'An error occurred while creating the clinic: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Update the specified clinic
     */
    public function update(Request $request, $id)
    {
        try {
            $clinic = Clinic::findOrFail($id);

            $validated = $request->validate([
                'clinic_name' => 'required|string|max:150',
                'clinic_location' => 'required|string|max:255',
                'admin_name' => 'nullable|string|max:100',
                'username' => 'nullable|string|max:100|unique:clinic,admin_username,' . $id,
                'password' => 'nullable|string|min:6',
                'status' => 'nullable|in:0,1',

                // Rooms, Departments, and Persons
                'rooms' => 'nullable|string',
                'departments' => 'nullable|string',
                'responsible_persons' => 'nullable|string',

                // Module Access Level
                'clinic_modules' => 'nullable|array',
                'clinic_modules.*' => 'nullable|string',
            ]);

            // Start database transaction
            DB::beginTransaction();

            $updateData = [
                'clinic_name' => $validated['clinic_name'],
                'clinic_location' => $validated['clinic_location'],
                'admin_name' => $validated['admin_name'] ?? null,
                'admin_username' => $validated['username'] ?? null,
                'name' => $validated['clinic_name'],
                'location' => $validated['clinic_location'],
                'status' => isset($validated['status']) ? $validated['status'] : $clinic->status,
                'modul_access' => json_encode($validated['clinic_modules'] ?? []),
            ];

            if (isset($validated['password']) && !empty($validated['password'])) {
                $updateData['admin_password'] = Hash::make($validated['password']);
            }

            $clinic->update($updateData);

            // Update Rooms - Delete old ones and create new ones
            if (isset($validated['rooms'])) {
                // Delete existing rooms for this clinic
                Room::where('clinic_id', $clinic->id)->delete();

                // Add new rooms
                $rooms = json_decode($validated['rooms'], true);
                if (is_array($rooms)) {
                    foreach ($rooms as $room) {
                        if (!empty($room['name'])) {
                            Room::create([
                                'clinic_id' => $clinic->id,
                                'name' => $room['name'],
                            ]);
                        }
                    }
                }
            }

            // Update Departments - Delete old ones and create new ones
            if (isset($validated['departments'])) {
                // Delete existing departments for this clinic
                HospitalDepartment::where('clinic_id', $clinic->id)->delete();

                // Add new departments
                $departments = json_decode($validated['departments'], true);
                if (is_array($departments)) {
                    foreach ($departments as $dept) {
                        if (!empty($dept['name'])) {
                            HospitalDepartment::create([
                                'clinic_id' => $clinic->id,
                                'name' => $dept['name'],
                            ]);
                        }
                    }
                }
            }

            // Update Responsible Persons - Delete old ones and create new ones
            if (isset($validated['responsible_persons'])) {
                // Delete existing persons for this clinic
                ResponsiblePerson::where('clinic_id', $clinic->id)->delete();

                // Add new persons
                $persons = json_decode($validated['responsible_persons'], true);
                if (is_array($persons)) {
                    foreach ($persons as $person) {
                        if (!empty($person['name']) && !empty($person['username'])) {
                            ResponsiblePerson::create([
                                'clinic_id' => $clinic->id,
                                'name' => $person['name'],
                                'username' => $person['username'],
                                'password' => Hash::make($person['password']),
                                'role' => $person['role'],
                                'access_level' => json_encode($person['access_level'] ?? []),
                            ]);
                        }
                    }
                }
            }

            // Commit transaction
            DB::commit();

            // Return JSON response for AJAX requests
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Clinic updated successfully!',
                    'clinic' => $clinic
                ]);
            }

            return redirect()->route('clinics')
                ->with('success', 'Clinic updated successfully!');

        } catch (\Illuminate\Validation\ValidationException $e) {
            DB::rollBack();
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => array_values($e->errors())[0][0] ?? 'Validation error',
                    'errors' => $e->errors()
                ], 422);
            }

            return redirect()->back()
                ->withErrors($e->errors())
                ->withInput();
        } catch (\Exception $e) {
            DB::rollBack();
            if ($request->expectsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'An error occurred while updating the clinic: ' . $e->getMessage()
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'An error occurred while updating the clinic: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Update clinic status
     */
    public function updateStatus(Request $request, $id)
    {
        try {
            $clinic = Clinic::findOrFail($id);

            $validated = $request->validate([
                'status' => 'required|in:0,1',
            ]);

            $clinic->update([
                'status' => $validated['status']
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Clinic status updated successfully!',
                'status' => $clinic->status
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified clinic
     */
    public function destroy($id)
    {
        try {
            $clinic = Clinic::findOrFail($id);
            $clinic->delete();

            return redirect()->route('clinics')
                ->with('success', 'Clinic deleted successfully!');

        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'An error occurred while deleting the clinic: ' . $e->getMessage());
        }
    }


    /**
     * Show the form for editing a clinic
     */
    public function edit($id)
    {
        try {
            $clinic = Clinic::with(['rooms', 'departments', 'responsiblePersons'])->findOrFail($id);

            // Get all master data for dropdowns
            $masterRooms = Room::whereNull('clinic_id')->orderBy('name', 'asc')->get();
            $masterDepartments = HospitalDepartment::whereNull('clinic_id')->orderBy('name', 'asc')->get();
            $masterPersons = ResponsiblePerson::whereNull('clinic_id')->orderBy('name', 'asc')->get();

            return view('addnewClinic', compact('clinic', 'masterRooms', 'masterDepartments', 'masterPersons'));
        } catch (\Exception $e) {
            return redirect()->route('clinics')
                ->with('error', 'Clinic not found');
        }
    }

    /**
     * Get paginated clinics data for AJAX requests
     */
    public function getClinicsPaginated(Request $request)
    {
        try {
            $perPage = $request->get('per_page', 10);
            $page = $request->get('page', 1);
            $search = $request->get('search', '');
            $filter = $request->get('filter', 'all');
            $sort = $request->get('sort', 'name-asc');

            $query = Clinic::query();

            // Apply search
            if ($search) {
                $query->where(function($q) use ($search) {
                    $q->where('clinic_name', 'like', "%{$search}%")
                      ->orWhere('clinic_location', 'like', "%{$search}%");
                });
            }

            // Apply filter
            if ($filter !== 'all') {
                $status = $filter === 'active' ? 1 : 0;
                $query->where('status', $status);
            }

            // Apply sorting
            list($column, $direction) = explode('-', $sort);
            $sortColumn = $column === 'name' ? 'clinic_name' : 'clinic_location';
            $query->orderBy($sortColumn, $direction);

            $clinics = $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $clinics->items(),
                'current_page' => $clinics->currentPage(),
                'last_page' => $clinics->lastPage(),
                'per_page' => $clinics->perPage(),
                'total' => $clinics->total(),
                'from' => $clinics->firstItem(),
                'to' => $clinics->lastItem(),
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'An error occurred: ' . $e->getMessage()
            ], 500);
        }
    }
}
