<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\AdminUser;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;

class AuthController extends Controller
{
    /**
     * Show the login form
     */
    public function showLogin()
    {
        return view('signin');
    }

    /**
     * Handle login request
     */
    public function login(Request $request)
    {
        // Validate the input
        $request->validate([
            'username' => 'required|string',
            'password' => 'required|string',
        ]);

        // Find the admin user by username
        $admin = AdminUser::where('username', $request->username)->first();

        // Check if user exists and password is correct
        if ($admin && Hash::check($request->password, $admin->password)) {
            // Store admin info in session
            Session::put('admin_id', $admin->id);
            Session::put('admin_name', $admin->name);
            Session::put('admin_username', $admin->username);
            Session::put('admin_role', $admin->role);
            Session::put('admin_access_level', $admin->access_level);
            Session::put('is_logged_in', true);

            // Redirect to dashboard
            return redirect()->route('dashboard')->with('success', 'Login successful!');
        }

        // If authentication fails
        return back()->withErrors([
            'login' => 'Invalid username or password.',
        ])->withInput($request->only('username'));
    }

    /**
     * Handle logout request
     */
    public function logout()
    {
        Session::flush();
        return redirect()->route('sign-in')->with('success', 'Logged out successfully!');
    }

    /**
     * Handle profile update request
     */
    public function updateProfile(Request $request)
    {
        // Validate the input
        $request->validate([
            'name' => 'required|string|max:100',
            'username' => 'required|string|max:100',
        ]);

        // Get the currently logged-in admin
        $adminId = Session::get('admin_id');
        if (!$adminId) {
            return response()->json([
                'message' => 'You are not logged in.'
            ], 401);
        }

        $admin = AdminUser::find($adminId);
        if (!$admin) {
            return response()->json([
                'message' => 'Admin user not found.'
            ], 404);
        }

        // Check if username is already taken by another user
        $existingAdmin = AdminUser::where('username', $request->username)
            ->where('id', '!=', $adminId)
            ->first();

        if ($existingAdmin) {
            return response()->json([
                'message' => 'Username is already taken.'
            ], 422);
        }

        // Update profile
        $admin->name = $request->name;
        $admin->username = $request->username;
        $admin->save();

        // Update session data
        Session::put('admin_name', $admin->name);
        Session::put('admin_username', $admin->username);

        return response()->json([
            'message' => 'Profile updated successfully!',
            'data' => [
                'name' => $admin->name,
                'username' => $admin->username
            ]
        ], 200);
    }

    /**
     * Handle password change request
     */
    public function changePassword(Request $request)
    {
        // Validate the input
        $request->validate([
            'current_password' => 'required|string',
            'new_password' => 'required|string|min:6',
            'new_password_confirmation' => 'required|string|same:new_password',
        ]);

        // Get the currently logged-in admin
        $adminId = Session::get('admin_id');
        if (!$adminId) {
            return response()->json([
                'message' => 'You are not logged in.'
            ], 401);
        }

        $admin = AdminUser::find($adminId);
        if (!$admin) {
            return response()->json([
                'message' => 'Admin user not found.'
            ], 404);
        }

        // Verify current password
        if (!Hash::check($request->current_password, $admin->password)) {
            return response()->json([
                'message' => 'Current password is incorrect.'
            ], 422);
        }

        // Update password
        $admin->password = Hash::make($request->new_password);
        $admin->save();

        return response()->json([
            'message' => 'Password changed successfully!'
        ], 200);
    }
}
