<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\AdminUser;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;

class AdminUserController extends Controller
{
    /**
     * Get all admin users (excluding Super Admin)
     */
    public function index()
    {
        $users = AdminUser::where('role', '!=', 'Super Admin')
                          ->orderBy('created_at', 'desc')
                          ->get();
        return response()->json($users);
    }

    /**
     * Store a new admin user
     */
    public function store(Request $request)
    {
        // Validate the input
        $validated = $request->validate([
            'name' => 'required|string|max:100',
            'username' => 'required|string|max:100|unique:admin_user,username',
            'password' => 'required|string|min:6',
            'role' => 'required|string|max:50',
            'access_level' => 'nullable|array',
        ]);

        // Create new admin user
        $user = new AdminUser();
        $user->name = $validated['name'];
        $user->username = $validated['username'];
        $user->password = Hash::make($validated['password']);
        $user->role = $validated['role'];
        $user->access_level = $validated['access_level'] ?? [];
        $user->save();

        return response()->json([
            'message' => 'User created successfully!',
            'user' => $user
        ], 201);
    }

    /**
     * Update an existing admin user
     */
    public function update(Request $request, $id)
    {
        $user = AdminUser::find($id);

        if (!$user) {
            return response()->json([
                'message' => 'User not found.'
            ], 404);
        }

        // Validate the input
        $validated = $request->validate([
            'name' => 'required|string|max:100',
            'username' => 'required|string|max:100|unique:admin_user,username,' . $id,
            'password' => 'nullable|string|min:6',
            'role' => 'required|string|max:50',
            'access_level' => 'nullable|array',
        ]);

        // Update user
        $user->name = $validated['name'];
        $user->username = $validated['username'];

        // Only update password if provided
        if (!empty($validated['password'])) {
            $user->password = Hash::make($validated['password']);
        }

        $user->role = $validated['role'];
        $user->access_level = $validated['access_level'] ?? [];
        $user->save();

        return response()->json([
            'message' => 'User updated successfully!',
            'user' => $user
        ], 200);
    }

    /**
     * Delete an admin user
     */
    public function destroy($id)
    {
        // Prevent deleting currently logged-in admin
        $currentAdminId = Session::get('admin_id');

        if ($currentAdminId == $id) {
            return response()->json([
                'message' => 'You cannot delete yourself!'
            ], 422);
        }

        $user = AdminUser::find($id);

        if (!$user) {
            return response()->json([
                'message' => 'User not found.'
            ], 404);
        }

        $user->delete();

        return response()->json([
            'message' => 'User deleted successfully!'
        ], 200);
    }
}
