<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Invoice;
use App\Models\Clinic;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use Barryvdh\DomPDF\Facade\Pdf;

class GenerateMonthlyInvoices extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'invoices:generate-monthly';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Generate monthly invoices for all active clinics on the 15th of each month';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting monthly invoice generation...');

        $currentDate = Carbon::now();
        $clinics = Clinic::where('status', 1)->get();
        $generatedCount = 0;
        $skippedCount = 0;

        foreach ($clinics as $clinic) {
            // Check if invoice already exists for this month
            $existingInvoice = Invoice::where('clinic_id', $clinic->id)
                ->whereYear('invoice_date', $currentDate->year)
                ->whereMonth('invoice_date', $currentDate->month)
                ->first();

            if ($existingInvoice) {
                $this->warn("Invoice already exists for clinic: {$clinic->name}");
                $skippedCount++;
                continue;
            }

            try {
                $invoice = Invoice::create([
                    'clinic_id' => $clinic->id,
                    'invoice_number' => $this->generateInvoiceNumber(),
                    'invoice_date' => $currentDate->toDateString(),
                    'amount' => $this->calculateInvoiceAmount($clinic->id),
                    'status' => 'pending'
                ]);

                // Generate PDF
                $pdfPath = $this->generateInvoicePDF($invoice);
                $invoice->update(['pdf_path' => $pdfPath]);

                $this->info("Invoice generated for clinic: {$clinic->name} - Invoice #: {$invoice->invoice_number}");
                $generatedCount++;
            } catch (\Exception $e) {
                $this->error("Failed to generate invoice for clinic: {$clinic->name}. Error: {$e->getMessage()}");
            }
        }

        $this->info("Invoice generation completed!");
        $this->info("Generated: {$generatedCount} invoices");
        $this->info("Skipped: {$skippedCount} invoices (already exist)");

        return Command::SUCCESS;
    }

    private function generateInvoiceNumber()
    {
        $prefix = 'INV';
        $year = date('Y');
        $month = date('m');

        $lastInvoice = Invoice::whereYear('invoice_date', $year)
            ->whereMonth('invoice_date', $month)
            ->orderBy('id', 'desc')
            ->first();

        $number = $lastInvoice ? (intval(substr($lastInvoice->invoice_number, -4)) + 1) : 1;

        return sprintf('%s-%s%s-%04d', $prefix, $year, $month, $number);
    }

    private function calculateInvoiceAmount($clinicId)
    {
        // Calculate based on waste bags or fixed monthly fee
        // For now, using a fixed amount of $500 per month
        return 500.00;
    }

    private function generateInvoicePDF($invoice)
    {
        $invoice->load('clinic');

        $pdf = Pdf::loadView('invoices.invoice-pdf', compact('invoice'));

        $fileName = "invoice_{$invoice->invoice_number}.pdf";
        $path = "invoices/{$fileName}";

        Storage::put($path, $pdf->output());

        return $path;
    }
}
